module net.BurtonRadons.dig.platform.canvas;


private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.platform.font;
private import net.BurtonRadons.dig.platform.frame;
private import net.BurtonRadons.dig.platform.windowsTheme;
private import net.BurtonRadons.dig.common.color;
private import net.BurtonRadons.dig.platform.windows;

//private import std.c.windows.windows;
//private import std.ctype;
//private import std.string;
//private import std.c.stdio; /* used for va_list */

/** A user-drawn object.  You are responsible for deleting the background.
  */

class Canvas : Control
{
    private const wchar [] digPlatformClassName = "digPlatformCanvas";
    
    /** Setup the window class for instances. */
    static this ()
    {
        digPlatformWindowClass.style = 0;
        digPlatformWindowClass.lpfnWndProc = &digPlatformWindowProc;
        digPlatformWindowClass.cbClsExtra = 0;
        digPlatformWindowClass.cbWndExtra = 0;
        digPlatformWindowClass.hInstance = hinst;
        digPlatformWindowClass.hCursor = std.c.windows.windows.LoadCursorA ((_HANDLE) 0, IDC_ARROW);
        digPlatformWindowClass.hbrBackground = (_HANDLE) 0;//(_HBRUSH) (COLOR_WINDOW + 1);
        digPlatformWindowClass.lpszMenuName = null;
        digPlatformWindowClass.lpszClassName = wtoStringz (digPlatformClassName);
        std.c.windows.windows.RegisterClassA (&digPlatformWindowClass);
    }

    /** Register itself in the parent and then create a window. */
    this (Control parent)
    {
        super (parent);
        digPlatformStyle |= std.c.windows.windows.WS_VISIBLE | std.c.windows.windows.WS_CHILD;// WS_OVERLAPPEDWINDOW;
        digPlatformHWNDCreate (0, digPlatformClassName, null, digPlatformStyle, (_HANDLE) 0);
        digPlatformDDC = digPlatformHDC;
    }

    /** Destroy the canvas. */
    ~this ()
    {
        digPlatformDonePaint ();

        HTHEME [] values = digPlatformThemes.values;
        for (int c; c < values.length; c ++)
            WindowsTheme.CloseThemeData (values [c]);
    }

    /** Start painting - only allowed in response to an @a onPaint message. */
    void beginPaint ()
    {
        if (digPlatformDoubleBuffer)
        {
            _RECT rect = digPlatformPainter.rcPaint;;

            digPlatformHDC = CreateCompatibleDC (digPlatformDDC);
            digPlatformBitmap = CreateCompatibleBitmap (digPlatformDDC, rect.right - rect.left, rect.bottom - rect.top);
            SelectObject (digPlatformHDC, digPlatformBitmap);
            SetViewportOrgEx (digPlatformHDC, -rect.left, -rect.top, null);
            std.c.windows.windows.BeginPaint (digPlatformHWND, &digPlatformPainter);
            std.c.windows.windows.SelectObject (digPlatformHDC, digPlatformFontSet);
        }
        else
            std.c.windows.windows.BeginPaint (digPlatformHWND, &digPlatformPainter);
        std.c.windows.windows.SetBkMode (digPlatformHDC, TRANSPARENT);
    }

    /** Finish painting. */
    void endPaint ()
    {
        if (digPlatformDoubleBuffer)
        {
            _RECT rect = digPlatformPainter.rcPaint;
            
            //BitBlt (digPlatformDDC, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, digPlatformHDC, 0, 0, SRCCOPY);
            BitBlt (digPlatformDDC, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, digPlatformHDC, rect.left, rect.top, SRCCOPY);
            DeleteDC (digPlatformHDC);
            DeleteObject (digPlatformBitmap);
            std.c.windows.windows.EndPaint (digPlatformHWND, &digPlatformPainter);
            digPlatformHDC = digPlatformDDC;
        }
        else
            std.c.windows.windows.EndPaint (digPlatformHWND, &digPlatformPainter);
        digPlatformDonePaint ();
    }

/** @name Text Printing
  */

/**@{*/

    /** Print out a line of text. */
    void textPrint (int x, int y, char [] string)
    {
        _RECT rect;

        rect.left = x;
        rect.top = y;
        rect.right = x + 31000;
        rect.bottom = y + 640;
        DrawTextA (digPlatformHDC, string, string.length, &rect, DT_EXPANDTABS | DT_NOPREFIX | DT_LEFT);
        //TextOutA (hdc, x, y, string, string.length);
    }

    /** Print out some formatted text. */
    extern (C)
    void textFormat (int x, int y, char [] format, ...)
    {
        textFormat (x, y, format, cast (std.c.stdio.va_list) (&format + 1));
    }

    /** Print out some formatted text. */
    void textFormat (int x, int y, char [] format, std.c.stdio.va_list args)
    {
        char [8192] buffer;
        int length;

        length = std.c.stdio.vsprintf (buffer, format, args);
        textPrint (x, y, buffer [0 .. length]);
    }

    /** Print text in a region, using ellipses at the start and end of
      * the text to remove what does not fit.  It centers these ellipses
      * on the mark, so a mark of zero always displays the start of the
      * text, a mark of string length always displays the end, and a mark
      * in the middle shows that area.  w is the width of the region.
      */
    void textPrintEllipses (int x, int y, int w, int mark, char [] string)
    {
        SIZE base;
        _RECT rect;
        int ewidth;

        /* Initialise. */
        rect.left = x;
        rect.top = y;
        rect.right = x + 6400;
        rect.bottom = y + 640;
        mark = imid (0, mark, string.length);

        /* Find out how wide the start and end are. */
        ewidth = digPlatformTextCharWidth ("...");

        /* Give up the ghost if it's way too small. */
        if (ewidth * 2 >= w)
        {
            DrawTextA (digPlatformHDC, "...", 3, &rect, DT_EXPANDTABS | DT_NOPREFIX | DT_LEFT);
            return;
        }

        char [] test = new char [string.length + 8];

        /* Build up text, trying each section. */
        for (int m; ; m ++)
        {
            if (m > string.length * 2)
            {
                DrawTextA (digPlatformHDC, string, string.length, &rect, DT_EXPANDTABS | DT_NOPREFIX | DT_LEFT);
                delete test;
                return;
            }

            if (digPlatformEllipseWidth (string, mark, m) > w)
            {
                char [] c = digPlatformEllipseArea (string, test, mark, m - 1);
                DrawTextA (digPlatformHDC, c, c.length, &rect, DT_EXPANDTABS | DT_NOPREFIX | DT_LEFT);
                delete test;
                return;
            }
        }
    }
    
    /** Return the width in pixels of ellipses-printed text.
      *
      * @param w The maximum width of the region.
      * @param mark The center to base ellipses around.
      * @param string The text to measure.
      */
      
    int textWidthEllipses (int w, int mark, char [] string)
    {
        SIZE base;
        _RECT rect;
        int ewidth;

        /* Initialise. */
        rect.left = 0;
        rect.top = 0;
        rect.right = 6400;
        rect.bottom = 640;
        mark = imid (0, mark, string.length);

        /* Find out how wide the start and end are. */
        ewidth = digPlatformTextCharWidth ("...");

        /* Give up the ghost if it's way too small. */
        if (ewidth * 2 >= w)
            return textWidth ("...");

        char [] test = new char [string.length + 8];

        /* Build up text, trying each section. */
        for (int m; ; m ++)
        {
            if (m > string.length * 2)
            {
                int result = textWidth (string);
                delete test;
                return result;
            }

            if (digPlatformEllipseWidth (string, mark, m) > w)
            {
                char [] c = digPlatformEllipseArea (string, test, mark, m - 1);
                int result;
                
                result = textWidth (c);
                delete test;
                return result;
            }
        }
    }

    /** Text alignment; can be '<' for left (default), '|' for center, '>' for right,
        '^' for vertical top (default), 'v' for vertical bottom, 'b' for vertical baseline. */
    void textAlign (char [] value)
    {
        int mask = 0;

             if (std.string.find (value, (char) 'b') >= 0) mask |= std.c.windows.windows.TA_BASELINE;
        else if (std.string.find (value, (char) 'v') >= 0) mask |= std.c.windows.windows.TA_BOTTOM;
        else if (std.string.find (value, (char) '^') >= 0) mask |= std.c.windows.windows.TA_TOP;
        else mask |= std.c.windows.windows.TA_TOP;

             if (std.string.find (value, (char) '<') >= 0) mask |= std.c.windows.windows.TA_LEFT;
        else if (std.string.find (value, (char) '|') >= 0) mask |= std.c.windows.windows.TA_CENTER;
        else if (std.string.find (value, (char) '>') >= 0) mask |= std.c.windows.windows.TA_RIGHT;
        else mask |= std.c.windows.windows.TA_LEFT;

        std.c.windows.windows.SetTextAlign (digPlatformHDC, mask);
    }
    
    /** Set the text color. */
    void textColor (int r, int g, int b)
    {
        std.c.windows.windows.SetTextColor (digPlatformHDC, digPlatformClampColor (r, g, b));
    }

    /** Set the text color. */
    void textColor (Color c)
    {
        textColor (c.r, c.g, c.b);
    }

    /** Get the width in pixels of a string of text. */
    int textWidth (char [] text)
    {
        wchar [] wtext = std.utf.toUTF16 (text);
        int result = GetTabbedTextExtentW (digPlatformHDC, wtext, wtext.length, 0, null);

        return result & 0xFFFF;
    }

    /** The height in pixels of the current font. */
    int textHeight ()
    {
        TEXTMETRIC metric;

        GetTextMetricsA (digPlatformHDC, &metric);
        return metric.tmHeight;
    }

/**@}*/

/** @name Pen Parameters
  */

/**@{*/

    /** Change the width of the pen in pixels. */
    void penWidth (int value)
    {
        digPlatformPenWidth = value;
        digPlatformSetPen ();
    }

    /** Set the style of the pen, one of:
      * 
      * \n @p null  - Invisible.
      * \n @p "---" - Solid (default).
      * \n @p "- -" - Dash.  The width is set to one.
      * \n @p "..." - Dotted.  The width is set to one.
      * \n @p "-.-" - Alternating dashes and dots.  The width is set to one.
      * \n @p "-.." - Alternating dashes and double-dots.  The width is set to one.
      */

    void penStyle (char value [])
    {
        int style;

        if (value === null) style = 0;
        else if (value == "---") style = 1;
        else if (value == "- -") style = 2;
        else if (value == "...") style = 3;
        else if (value == "-.-") style = 4;
        else if (value == "-..") style = 5;
        else style = 1;

        digPlatformPenStyle = style;
        digPlatformSetPen ();
    }
    
    /** Set to no pen. */
    void penClear ()
    {
        penStyle (null);
    }

    /** Set the pen color. */
    void penColor (int r, int g, int b)
    {
        digPlatformPenColor = digPlatformClampColor (r, g, b);
        digPlatformSetPen ();
    }

    /** Set the pen color. */
    void penColor (Color c)
    {
        penColor (c.r, c.g, c.b);
    }

/**@}*/


/** @name Brush Parameters
  */

/**@{*/

    /** Set to a clear brush. */
    void brushClear ()
    {
        digPlatformSetBorrowedBrush (GetStockObject (HOLLOW_BRUSH));
    }

    /** Set to a solid color brush. */
    void brushColor (int r, int g, int b)
    {
        digPlatformSetBrush (CreateSolidBrush (digPlatformClampColor (r, g, b)));
    }

    /** Set to a solid color brush. */
    void brushColor (Color c)
    {
        brushColor (c.r, c.g, c.b);
    }

/**@}*/

/** @name Drawing Operations
  */

/**@{*/

    /** Draw a circle using the pen in the outline and the brush to fill. */
    void circle (int x, int y, int radius)
    {
        circle (x, y, radius, radius);
    }

    /** Draw a centered ellipse using the pen in the outline and the brush to fill. */
    void circle (int x, int y, int width, int height)
    {
        ellipse (x - width / 2, y - height / 2, x + (width + 1) / 2, y + (height + 1) / 2);
    }

    /** Clear the whole region to a std.math.single color. */
    void clear (int r, int g, int b)
    {
        _HBRUSH obrush = digPlatformBrush;
        bit obrushBorrowed = digPlatformBrushBorrowed;
        int style = digPlatformPenStyle;
        
        digPlatformBrush = (_HANDLE) 0;
        penStyle (null);
        brushColor (r, g, b);
        Rectangle (digPlatformHDC, 0, 0, width () + 1, height () + 1);
        digPlatformSetBrush (obrush);
        digPlatformBrushBorrowed = obrushBorrowed;
        digPlatformPenStyle = style;
        digPlatformSetPen ();
    }

    /** Clear to a std.math.single color. */
    void clear (Color c)
    {
        clear (c.r, c.g, c.b);
    }

    /** Draw an ellipse using the pen to outline and the brush to fill. */
    void ellipse (int sx, int sy, int ex, int ey)
    {
        Ellipse (digPlatformHDC, sx, sy, ex, ey);
    }

    /** Draw a line using the current pen. */
    void line (int sx, int sy, int ex, int ey)
    {
        _POINT [2] points;

        points [0].x = sx;
        points [0].y = sy;
        points [1].x = ex;
        points [1].y = ey;
        Polyline (digPlatformHDC, points, 2);
    }

    /** Draw a series of lines using the current pen; each point is an x, y pair. */
    void line (int [] points)
    {
        Polyline (digPlatformHDC, (_POINT *) (int *) points, points.length / 2);
    }

    /** Draw a polygon using the current pen and fill it with the current brush; each point is an x, y pair. */
    void polygon (int [] points)
    {
        Polygon (digPlatformHDC, (_POINT *) (int *) points, points.length / 2);
    }

    /** Draw a triangle using the current pen and fill it with the current brush. */
    void polygon (int ax, int ay, int bx, int by, int cx, int cy)
    {
        int [6] p;

        p [0] = ax; p [1] = ay;
        p [2] = bx; p [3] = by;
        p [4] = cx; p [5] = cy;
        Polygon (digPlatformHDC, (_POINT *) (int *) p, p.length / 2);
    }

    /** Draw a rounded rectangle outlined with the current pen and filled using the current brush. */
    void roundRect (int left, int top, int right, int bottom, int diameter)
    {
        roundRect (left, top, right, bottom, diameter, diameter);
    }

    /** Draw a rounded rectangle outlined with the current pen and filled using the current brush. */
    void roundRect (int left, int top, int right, int bottom, int width, int height)
    {
        RoundRect (digPlatformHDC, left, top, right, bottom, width, height);
    }

    /** Draw a rectangle outlined with the current pen and filled using the current brush. */
    void rect (int left, int top, int right, int bottom)
    {
        Rectangle (digPlatformHDC, left, top, right, bottom);
    }

/**@}*/

/** @name Primitive Emulators
  * These methods draw primitives as they would be drawn by other 
  * OS/skin-specific controls.
  */

/**@{*/

    /** Draw a listbox column header item. */
    void listboxColumnHeaderDraw (int sx, int sy, int ex, int ey, bit highlight, bit pressed, char [] name)
    {
        HTHEME theme;

        if ((theme = digPlatformGetTheme ("header")) != (_HANDLE) 0)
        {
            int part = WindowsTheme.HP_HEADERITEM;
            int state = pressed ? WindowsTheme.HIS_PRESSED : highlight ? WindowsTheme.HIS_HOT : WindowsTheme.HIS_NORMAL;
            wchar [] wname = toWStringz (name);
            MARGINS margins;
            LOGFONT font;

            WindowsTheme.DrawThemeBackground (theme, digPlatformHDC, part, state, digPlatformGetRectp (sx, sy, ex, ey), null);

            WindowsTheme.GetThemeMargins (theme, digPlatformHDC, part, state, WindowsTheme.TMT_CONTENTMARGINS, digPlatformGetRectp (sx, sy, ex, ey), &margins);

            WindowsTheme.GetThemeFont (theme, digPlatformHDC, part, state, WindowsTheme.TMT_CAPTIONFONT, &font);

            _HANDLE fontHandle;
            SelectObject (digPlatformHDC, fontHandle = CreateFontIndirectA (&font));
            WindowsTheme.DrawThemeText (theme, digPlatformHDC, part, state, wname, -1, DT_LEFT, 0, digPlatformGetRectp (sx + margins.cxLeftWidth, sy + margins.cyTopHeight, ex - margins.cxRightWidth, ey - margins.cyBottomHeight));
            SelectObject (digPlatformHDC, digPlatformFontSet);
            DeleteObject (fontHandle);
            delete wname;
        }
        else
        {
            int edge, mode;

            if (pressed)
                edge = BDR_SUNKENINNER | BDR_SUNKENOUTER;
            else
                edge = BDR_RAISEDINNER | BDR_RAISEDOUTER;

            mode = BF_RECT;
            DrawEdge (digPlatformHDC, digPlatformGetRectp (sx, sy, ex, ey), edge, mode);
            textPrintEllipses (sx + 4, sy + 2, ex - 4, 0, name);
        }
    }

    /** Draw the listbox column header sorting arrow. */
    void listboxColumnArrowDraw (int sx, int sy, int ex, int ey, bit down)
    {
        penClear ();
        brushColor (AColor (164, 164, 164));

        int ax, ay, bx, by, cx, cy;

        ax = (ex - sx) / 2 + sx; ay = sy + 6;
        bx = sx; by = ey - 9; bx = ax - (by - ay);
        cx = ex; cy = ey - 9; cx = ax + (by - ay) - 1;

        if (!down)
        {
            int ty = by;
            by = ay;
            cy = ay;
            ay = ty;
            bx ++;
            cx ++;
        }

        polygon (bx, by + 1, ax, ay + 1, cx, cy + 1);
    }

    /** Draw a listbox item. */
    void listboxItemDraw (int sx, int sy, int ex, int ey, bit sortColumn)
    {
        Color color = digPlatformColorFromCOLORREF (GetSysColor (COLOR_WINDOW));

        if (sortColumn)
            color = AColor (color.r * 0.97, color.g * 0.97, color.b * 0.97);

        brushColor (color);
        penClear ();
        rect (sx, sy, ex + 1, ey + 1);
    }

    /** Draw a spinner button. */
    void spinnerDraw (int sx, int sy, int ex, int ey, bit upPressed, bit downPressed, bit upHighlight, bit downHighlight)
    {
        int mid = ((ey - sy) + 1) / 2 + sy;
        HTHEME theme;

        if ((theme = digPlatformGetTheme ("spin")) != (_HANDLE) 0)
        {
            alias WindowsTheme w;
            int state;
        
            state = upPressed ? w.UPS_PRESSED : upHighlight ? w.UPS_HOT : w.UPS_NORMAL;
            w.DrawThemeBackground (theme, digPlatformHDC, w.SPNP_UP, state, digPlatformGetRectp (sx, sy, ex, mid), null);

            state = downPressed ? w.DNS_PRESSED : downHighlight ? w.DNS_HOT : w.DNS_NORMAL;
            w.DrawThemeBackground (theme, digPlatformHDC, w.SPNP_DOWN, state, digPlatformGetRectp (sx, mid, ex, ey), null);
        }
        else
        {
            buttonDraw (sx, sy, ex, mid, upPressed, false);
            buttonDraw (sx, mid, ex, ey, downPressed, false);

            penClear ();
            brushColor (Color.Black);

            int ax, ay, bx, by, cx, cy;

            ax = width () / 2 - 1; ay = 3;
            bx = 3 - 1; by = height () / 2 - 3;
            cx = width () - 3 - 1; cy = by;

            if (ay <= by && ax >= bx)
                polygon (ax, ay, bx, by, cx, cy);

            ay = height () - 3 - 1;
            by = mid + 3;
            cy = by;
            bx ++;

            if (ay >= by && ax >= bx)
                polygon (ax, ay, bx, by, cx, cy);
        }
    }

    /** Draw a button border. */
    void buttonDraw (int sx, int sy, int ex, int ey, bit down, bit focus)
    {
        HTHEME theme;
        _UINT edge;

        if ((theme = digPlatformGetTheme ("button")) != (_HANDLE) 0)
        {
            alias WindowsTheme w;
            int state;

            state = down ? w.PBS_PRESSED : w.PBS_NORMAL;
            w.DrawThemeBackground (theme, digPlatformHDC, w.BP_PUSHBUTTON, state, digPlatformGetRectp (sx, sy, ex, ey), null);
        }
        else
        {
            edge = down ? EDGE_SUNKEN : EDGE_RAISED;
            DrawEdge (digPlatformHDC, digPlatformGetRectp (sx, sy, ex, ey),
                      edge, BF_RECT | BF_MIDDLE);

            if (focus)
                DrawFocusRect (digPlatformHDC, digPlatformGetRectp (sx + 3, sy + 3, ex - 3, ey - 3));
        }
    }

    /** Get the button border size. */
    void buttonBorder (out int width, out int height)
    {
        width = 6;
        height = 6;
    }

    /** Get the button client region from the input. */
    void buttonClientRegion (inout int sx, inout int sy, inout int ex, inout int ey)
    {
        sx += 3;
        sy += 3;
        ex -= 3;
        ey -= 3;
    }

    /** Add the content offset when the button is pressed. */
    void buttonPressedOffset (inout int x, inout int y)
    {
        x += 1;
        y += 1;
    }

    /** Draw a group box. */
    void groupboxDraw (int sx, int sy, int ex, int ey, char [] caption)
    {
        HTHEME theme;
        int h = textHeight ();

        if ((theme = digPlatformGetTheme ("button")) != (_HANDLE) 0)
        {
            alias WindowsTheme w;
            wchar [] string = toWStringz (caption);
            _RECT textrect;
            int part = w.BP_GROUPBOX, state;
            int textflags = DT_LEFT;
            MARGINS margins;
            _POINT point;

            state = w.GBS_NORMAL;
            w.DrawThemeBackground (theme, digPlatformHDC, part, state, digPlatformGetRectp (sx, sy + h / 2, ex, ey), null);
            
            brushColor (backgroundColor ());
            penClear ();
            rect (sx + 6, sy, sx + 12 + textWidth (caption), sy + h);

            w.DrawThemeText (theme, digPlatformHDC, part, state, string, string.length, textflags, 0, digPlatformGetRectp (sx + 8, sy, ex, ey));
            delete string;
        }
        else
        {
            DrawEdge (digPlatformHDC, digPlatformGetRectp (sx, sy + h / 2, ex, ey), EDGE_ETCHED, BF_RECT);

            brushColor (backgroundColor ());
            penClear ();
            rect (sx + 6, sy, sx + 12 + textWidth (caption), sy + h);
            textColor (0, 0, 0);
            textPrint (sx + 8, sy, caption);
        }
    }

    /** Get the group box border size. */
    void groupboxBorder (out int width, out int height)
    {
        width = 10;
        height = 3;
    }

    /** Get the groupbox client region from the input. */
    void groupboxClientRegion (inout int sx, inout int sy, inout int ex, inout int ey)
    {
        sx += 10;
        sy += 3 + textHeight ();
        ex -= 10;
        ey -= 3;
    }

    /** The type of boxes that can be drawn with drawbox.
      */
    enum DB
    {
        ThinSunken, /**< A thin inset border, as seen with Windows status bars. */
        ThinRaised, /**< A thin raised border. */
        Focus,      /**< A keyboard focus border. */
    }
    
    /** Draw a box using a given type. */
    void drawbox (int sx, int sy, int ex, int ey, DB type, bit filled)
    {
        _UINT edge, mode;

        if (type == DB.ThinSunken)
        {
            edge = BDR_SUNKENINNER;
            mode = BF_RECT;
        }
        else if (type == DB.ThinRaised)
        {
            edge = BDR_RAISEDINNER;
            mode = BF_RECT;
        }
        else if (type == DB.Focus)
        {
            if (filled)
                DrawEdge (digPlatformHDC, digPlatformGetRectp (sx, sy, ex, ey), 0, BF_MIDDLE);
            DrawFocusRect (digPlatformHDC, digPlatformGetRectp (sx, sy, ex, ey));
            return;
        }

        if (filled)
            mode |= BF_MIDDLE;

        DrawEdge (digPlatformHDC, digPlatformGetRectp (sx, sy, ex, ey), edge, mode);
    }

    /** Get the border size of the drawbox. */
    void drawboxBorder (out int width, out int height, DB type)
    {
        if (type == DB.ThinSunken || type == DB.ThinRaised)
        {
            width = 2;
            height = 2;
        }

        if (type == DB.Focus)
            width = 1, height = 1;
    }

    /** Get the client region of a certain type. */
    void drawboxClientRegion (inout int sx, inout int sy, inout int ex, inout int ey, DB type)
    {
        if (type == DB.ThinSunken || type == DB.ThinRaised)
        {
            sx += 2;
            sy += 2;
            ex -= 2;
            ey -= 2;
        }
        else if (type == DB.Focus)
            sx += 1, sy += 1, ex -= 1, ey -= 1;
    }

    /** Draw a status bar background. */
    void statusbarBackgroundDraw (int sx, int sy, int ex, int ey)
    {
        HTHEME theme;

        if ((theme = digPlatformGetTheme ("status")) != (_HANDLE) 0)
        {
            alias WindowsTheme w;
            int part = 4; /* Yes, really, it's not documented. */
            int state = 0;

            w.DrawThemeBackground (theme, digPlatformHDC, part, state, digPlatformGetRectp (sx, sy, ex, ey), null);
        }
        else
        {
            penClear ();
            brushColor (backgroundColor ());
            rect (sx, sy, ex, ey);
            drawbox (0, 0, width (), height (), DB.ThinRaised, false);
        }
    }

    /** Draw a status bar pane, which either surrounds or delimits entries in the status bar. */
    void statusbarPaneDraw (int sx, int sy, int ex, int ey)
    {
        HTHEME theme;

        if ((theme = digPlatformGetTheme ("status")) != (_HANDLE) 0)
        {
            alias WindowsTheme w;
            int part = w.SP_PANE;
            int state = 0;

            w.DrawThemeBackground (theme, digPlatformHDC, part, state, digPlatformGetRectp (sx - 3, sy + 1, sx - 1, ey), null);
        }
        else
            drawbox (sx, sy + 2, ex, ey - 1, DB.ThinSunken, true);
    }
    
    /** Draw a tree box control cross.
      *
      * @param x The starting horizontal coordinate.
      * @param y The starting vertical coordinate.
      * @param opened Whether the cross is opened (true) or closed (false).
      */
    void treeBoxCrossDraw (int x, int y, bit opened)
    {
        HTHEME theme;
        
        if ((theme = digPlatformGetTheme ("TreeView")) != (_HANDLE) 0)
        {
            alias WindowsTheme W;
            int part = W.TVP_GLYPH;
            int state = opened ? W.GLPS_OPENED : W.GLPS_CLOSED;
            int ex = x + treeBoxCrossWidth (opened);
            int ey = y + treeBoxCrossHeight (opened);
            
            W.DrawThemeBackground (theme, digPlatformHDC, part, state, digPlatformGetRectp (x, y, ex, ey), null);
        }
    }
    
    /** Retrieve the size in pixels of the control cross. */
    void treeBoxCrossSize (out int width, out int height, bit opened)
    {
        HTHEME theme;
        
        if ((theme = digPlatformGetTheme ("TreeView")) != (_HANDLE) 0)
        {
            alias WindowsTheme W;
            int part = W.TVP_GLYPH;
            int state = opened ? W.GLPS_OPENED : W.GLPS_CLOSED;
            SIZE result;
            
            W.GetThemePartSize (theme, digPlatformHDC, part, state, null, W.THEMESIZE.TS_TRUE, &result);
            width = result.cx;
            height = result.cy;
        }
    }
    
    /** Return the width in pixels of the control cross. */
    int treeBoxCrossWidth (bit opened)
    {
        int width, height;
        
        treeBoxCrossSize (width, height, opened);
        return width;
    }
    
    /** Return the height in pixels of the control cross. */
    int treeBoxCrossHeight (bit opened)
    {
        int width, height;
        
        treeBoxCrossSize (width, height, opened);
        return height;
    }

/**@}*/

    /** Get the current double-buffering value. */
    bit doubleBuffer ()
    {
        return digPlatformDoubleBuffer;
    }

    /** Set double-buffering.  When true (the default), painting first goes to
      * a temporary bitmap that is then blitted on endPaint.  This removes
      * flicker, but is more expensive than std.math.single-buffering.
      */

    void doubleBuffer (bit value)
    {
        digPlatformDoubleBuffer = value;
    }

/+
#ifdef DOXYGEN_JUST_WOULDNT_UNDERSTAND
+/

    static _WNDCLASS digPlatformWindowClass;
    _PAINTSTRUCT digPlatformPainter;
    bit digPlatformPainting;

    _HPEN digPlatformPen; /* Current pen (0 for none) */
    int digPlatformPenStyle = 1; /* Current pen style */
    int digPlatformPenWidth = 1; /* Current pen width */
    _COLORREF digPlatformPenColor; /* Current pen color */

    _HBRUSH digPlatformBrush; /* Current brush (0 for none/white) */
    bit digPlatformBrushBorrowed; /* Borrowed brush; don't delete it */
    bit digPlatformDoubleBuffer = true; /* Double-buffering has been enabled */

    _HDC digPlatformDDC;
    _HBITMAP digPlatformBitmap;
    
    HTHEME [char []] digPlatformThemes;
    
    override void digPlatformOnFontChange ()
    {
        _HFONT handle = digPlatformFont.digPlatformGetHFONT ();
        
        SendMessageA (digPlatformHWND, WM_SETFONT, (uint) handle, !digPlatformPainting);
        SelectObject (digPlatformHDC, handle);
        GetCharWidth32A (digPlatformHDC, 0, 255, digPlatformCharWidths);
        if (digPlatformFontSet)
            DeleteObject (digPlatformFontSet);
        digPlatformFontSet = handle;
    }
    
    int [256] digPlatformCharWidths;

    int digPlatformTextCharWidth (char [] string)
    {
        int w;

        for (char *s = string, e = s + string.length; s < e; s ++)
            w += digPlatformCharWidths [*s];

        return w;
    }

    char [] digPlatformEllipseArea (char [] string, char [] test, int mark, int w)
    {
        if (w < 0)
            return "...";

        int s = imid (0, mark - w / 2, string.length);
        int e = imid (0, mark + (w + 1) / 2, string.length);
        int l;

        l = std.c.stdio.sprintf (test, "%.*s%.*s%.*s", 
            s ? (char []) "..." : (char []) "",
            string [s .. e],
            e != string.length ? (char []) "..." : (char []) "");

        return test [0 .. l];
    }

    int digPlatformEllipseWidth (char [] string, int mark, int w)
    {
        int b = digPlatformTextCharWidth ("...");

        if (w < 0)
            return b;

        int s = imid (0, mark - w / 2, string.length);
        int e = imid (0, mark + (w + 1) / 2, string.length);
        int l;

        if (s) l += b;
        if (e != string.length) l += b;
        l += digPlatformTextCharWidth (string [s .. e]);
        return l;
    }

    HTHEME digPlatformGetTheme (char [] name)
    {
        if (name in digPlatformThemes)
            return digPlatformThemes [name];
        if (WindowsTheme.OpenThemeData === null)
            return (_HANDLE) 0;
        digPlatformThemes [name] = WindowsTheme.OpenThemeData (digPlatformHWND, toWStringz (name));
        return digPlatformThemes [name];
    }
    
    void digPlatformRecreate ()
    {
        super.digPlatformRecreate ();
        digPlatformDDC = digPlatformHDC;
    }

    _RECT digPlatformGetRect (int left, int top, int right, int bottom)
    {
        _RECT r;

        r.left = left;
        r.top = top;
        r.right = right;
        r.bottom = bottom;
        return r;
    }

    _LPRECT digPlatformGetRectp (int left, int top, int right, int bottom)
    {
        static _RECT r;

        r = digPlatformGetRect (left, top, right, bottom);
        return &r;
    }

    int digPlatformClampColor (inout int r, inout int g, inout int b)
    {
        r = r < 0 ? 0 : r > 255 ? 255 : r;
        g = g < 0 ? 0 : g > 255 ? 255 : g;
        b = b < 0 ? 0 : b > 255 ? 255 : b;
        return r | (g << 8) | (b << 16);
    }

    /* Delete some crap */
    void digPlatformDonePaint ()
    {
        if (digPlatformBrush && !digPlatformBrushBorrowed)
            DeleteObject (digPlatformBrush);
        if (digPlatformPen != (_HANDLE) 0)
            DeleteObject (digPlatformPen);
        digPlatformBrush = (_HANDLE) 0;
        digPlatformPen = (_HANDLE) 0;
    }

    /* Delete the previous pen and setup a new one */
    void digPlatformSetPen ()
    {
        _HPEN npen;
        int width = digPlatformPenWidth;
        int style;

        switch (digPlatformPenStyle)
        {
            case 0: style = PS_NULL; break;
            case 1: style = PS_SOLID; break;
            case 2: style = PS_DASH; width = 1; break;
            case 3: style = PS_DOT; width = 1; break;
            case 4: style = PS_DASHDOT; width = 1; break;
            case 5: style = PS_DASHDOTDOT; width = 1; break;
            default: style = PS_SOLID; break;
        }

        npen = CreatePen (style, width, digPlatformPenColor);
        SelectObject (digPlatformHDC, npen);
        if (digPlatformPen)
            DeleteObject (digPlatformPen);
        digPlatformPen = npen;
    }

    /* Change the brush and delete the old one. */
    void digPlatformSetBrush (_HBRUSH brush)
    {
        SelectObject (digPlatformHDC, brush);
        if (digPlatformBrush && !digPlatformBrushBorrowed)
            DeleteObject (digPlatformBrush);
        digPlatformBrush = brush;
        digPlatformBrushBorrowed = false;
    }

    /* Change the brush to a borrowed one */
    void digPlatformSetBorrowedBrush (_HBRUSH brush)
    {
        digPlatformSetBrush (brush);
        digPlatformBrushBorrowed = true;
    }

    static extern (Windows)
    _LRESULT digPlatformWindowProc (_HWND hwnd, _UINT message, _WPARAM wParam, _LPARAM lParam)
    {
        Control control = digPlatformHWNDToControl [hwnd];
        Canvas canvas = (Canvas) control;
        int result = 0;
        Event event;

        switch (message)
        {
            case WM_PAINT:
            {
                _RECT rect;
                
                if (GetUpdateRect (control.digPlatformHWND, &rect, 0))
                {
                    event.left = rect.left;
                    event.top = rect.top;
                    event.right = rect.right;
                    event.bottom = rect.bottom;
                    canvas.digPlatformPainter.hdc = canvas.digPlatformHDC;
                    canvas.digPlatformPainter.rcPaint = rect;
                }
                else
                    goto fallback;
                canvas.digPlatformPainting = true;
                control.onPaint.notify (event);
                canvas.digPlatformPainting = false;
                break;
            }

            default:
            fallback:
                return Frame.digPlatformWindowProc (hwnd, message, wParam, lParam);
        }

        return result;
    }

    override void digPlatformChildMoved (Control control)
    {
        digPlatformGridfit ();
        digPlatformMoved ();
    }

/+
#endif
+/
}
